"""Stefan Spence 29.03.21
Investigate the effect of pulse shaping on Raman sideband transition linewidth
I. Boradjiev and N. Vitanov PHYSICAL REVIEW A 88, 013402 (2013)"""
import matplotlib.pyplot as plt
import numpy as np
import sys
from scipy.signal import tukey
import os
os.chdir(os.path.dirname(__file__))
sys.path.append('../..')
from fitandgraph import fit, c, hbar, kB, h
from qutip import *

w  = 120*2*np.pi # trap freq 2pi kHz
eta = (2.07/w*2*np.pi)**0.5 # LD parameter
T = 1   # Temp uK
N_ph = 10  # number of harmonic levels in bnasis
psi0 = tensor(fock_dm(N_ph, 0), fock_dm(2,0)) # initial state
f = fit()

rabis = np.linspace(w/10,w,100) # Rabi freq 2pi kHz
psquare = np.zeros(len(rabis))
pblack  = np.zeros(len(rabis))
ptukey  = np.zeros(len(rabis))

for i, Or in enumerate(rabis):
    dur = np.pi/Or/eta # duration of square pulse
    tsquare = np.linspace(0,dur,200)
    det = - w + Or**2/2./w # 2-photon detuning
    Hb, HR, _ = f.RamanHamiltonians(w, eta, Or, det, T, N_ph)
    square = mesolve([Hb, HR], psi0, tsquare, e_ops=tensor(fock_dm(N_ph,0), fock_dm(2,0)))
    psquare[i] = 1 - square.expect[0][-1]

    tblack = tsquare#/0.35875
    tscaled = tblack / max(tblack)
    HRcb = 0.35875 - 0.48829*np.cos(2*np.pi*tscaled) +0.14128*np.cos(4*np.pi*tscaled) - 0.01168*np.cos(6*np.pi*tscaled)
    det = - w + 0.3101328*Or**2/w # 2-photon detuning
    Hb, HR, _ = f.RamanHamiltonians(w, eta, Or*len(tblack)/np.sum(HRcb), det, T, N_ph)
    black = mesolve([Hb, [HR, HRcb]], psi0, tblack, e_ops=tensor(fock_dm(N_ph,0), fock_dm(2,0)))
    pblack[i] = 1 - black.expect[0][-1]

    ttukey = tsquare#*1.5 # equal area to square pulse
    HRct = tukey(len(ttukey), 2/3.)
    # HRct = tukey(len(ttukey), 2/3.)**0.5
    det = - w + 0.40230667*Or**2/w # 2-photon detuning
    Hb, HR, _ = f.RamanHamiltonians(w, eta, Or*len(ttukey)/np.sum(HRct), det, T, N_ph)
    tuke = mesolve([Hb, [HR, HRct]], psi0, ttukey, e_ops=tensor(fock_dm(N_ph,0), fock_dm(2,0)))
    ptukey[i] = 1 - tuke.expect[0][-1]
    
    print(i, end=' ')



rabis /= 2*np.pi

#%% plot results
fontsize = 18
plt.rc('axes', labelsize=fontsize+2)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=fontsize)    # fontsize of the tick labels
plt.rc('ytick', labelsize=fontsize)    # fontsize of the tick labels
plt.rc('legend', fontsize=fontsize)    # legend fontsize

filename = r'{}_probs.npy'
label = ['axial25Rb', 'radial100Rb'][1]
psquare, ptukey, pblack = np.load(filename.format(label))
if '25' in label:
    rabis = np.linspace(2.5,25,100)
else: rabis = np.linspace(10,100,100)
plt.figure()
plt.semilogy(rabis, psquare, label='Square', color='C2')
plt.semilogy(rabis, pblack, label='Blackman-Harris', color='C0')
plt.semilogy(rabis, ptukey, label='$\sqrt{Tukey}$', color='C4')  # 
ymin = min(plt.ylim())
plt.xlabel('Rabi Frequency (kHz)')
plt.ylabel(r'$1 - P(n=0)$')
# plt.legend() # loc=upper right
plt.xlim(min(rabis), max(rabis))
plt.ylim(ymin, 1)
plt.show()
# np.savetxt('Fig3c_data.csv', np.array([rabis, psquare, ptukey, pblack]).T, delimiter=',', 
#    header='Mean Rabi Frequency (kHz),Probability of excitation with square pulse,Probability of excitation with Tukey pulse,Probability of excitation with Blackman-Harris pulse')
